import { atom, computed, reactor, transact } from "@tldraw/state";
import {
  WeakCache,
  assert,
  filterEntries,
  getOwnProperty,
  objectMapEntries,
  objectMapFromEntries,
  objectMapKeys,
  objectMapValues,
  throttleToNextFrame
} from "@tldraw/utils";
import { nanoid } from "nanoid";
import { squashRecordDiffs } from "./RecordsDiff.mjs";
import { StoreQueries } from "./StoreQueries.mjs";
import { StoreSideEffects } from "./StoreSideEffects.mjs";
import { devFreeze } from "./devFreeze.mjs";
class Store {
  /**
   * The random id of the store.
   */
  id;
  /**
   * An atom containing the store's atoms.
   *
   * @internal
   * @readonly
   */
  atoms = atom("store_atoms", {});
  /**
   * An atom containing the store's history.
   *
   * @public
   * @readonly
   */
  history = atom("history", 0, {
    historyLength: 1e3
  });
  /**
   * A StoreQueries instance for this store.
   *
   * @public
   * @readonly
   */
  query = new StoreQueries(this.atoms, this.history);
  /**
   * A set containing listeners that have been added to this store.
   *
   * @internal
   */
  listeners = /* @__PURE__ */ new Set();
  /**
   * An array of history entries that have not yet been flushed.
   *
   * @internal
   */
  historyAccumulator = new HistoryAccumulator();
  /**
   * A reactor that responds to changes to the history by squashing the accumulated history and
   * notifying listeners of the changes.
   *
   * @internal
   */
  historyReactor;
  /**
   * Function to dispose of any in-flight timeouts.
   *
   * @internal
   */
  cancelHistoryReactor = () => {
  };
  schema;
  props;
  scopedTypes;
  sideEffects = new StoreSideEffects(this);
  constructor(config) {
    const { initialData, schema, id } = config;
    this.id = id ?? nanoid();
    this.schema = schema;
    this.props = config.props;
    if (initialData) {
      this.atoms.set(
        objectMapFromEntries(
          objectMapEntries(initialData).map(([id2, record]) => [
            id2,
            atom(
              "atom:" + id2,
              devFreeze(this.schema.validateRecord(this, record, "initialize", null))
            )
          ])
        )
      );
    }
    this.historyReactor = reactor(
      "Store.historyReactor",
      () => {
        this.history.get();
        this._flushHistory();
      },
      { scheduleEffect: (cb) => this.cancelHistoryReactor = throttleToNextFrame(cb) }
    );
    this.scopedTypes = {
      document: new Set(
        objectMapValues(this.schema.types).filter((t) => t.scope === "document").map((t) => t.typeName)
      ),
      session: new Set(
        objectMapValues(this.schema.types).filter((t) => t.scope === "session").map((t) => t.typeName)
      ),
      presence: new Set(
        objectMapValues(this.schema.types).filter((t) => t.scope === "presence").map((t) => t.typeName)
      )
    };
  }
  _flushHistory() {
    if (this.historyAccumulator.hasChanges()) {
      const entries = this.historyAccumulator.flush();
      for (const { changes, source } of entries) {
        let instanceChanges = null;
        let documentChanges = null;
        let presenceChanges = null;
        for (const { onHistory, filters } of this.listeners) {
          if (filters.source !== "all" && filters.source !== source) {
            continue;
          }
          if (filters.scope !== "all") {
            if (filters.scope === "document") {
              documentChanges ??= this.filterChangesByScope(changes, "document");
              if (!documentChanges) continue;
              onHistory({ changes: documentChanges, source });
            } else if (filters.scope === "session") {
              instanceChanges ??= this.filterChangesByScope(changes, "session");
              if (!instanceChanges) continue;
              onHistory({ changes: instanceChanges, source });
            } else {
              presenceChanges ??= this.filterChangesByScope(changes, "presence");
              if (!presenceChanges) continue;
              onHistory({ changes: presenceChanges, source });
            }
          } else {
            onHistory({ changes, source });
          }
        }
      }
    }
  }
  dispose() {
    this.cancelHistoryReactor();
  }
  /**
   * Filters out non-document changes from a diff. Returns null if there are no changes left.
   * @param change - the records diff
   * @returns
   */
  filterChangesByScope(change, scope) {
    const result = {
      added: filterEntries(change.added, (_, r) => this.scopedTypes[scope].has(r.typeName)),
      updated: filterEntries(change.updated, (_, r) => this.scopedTypes[scope].has(r[1].typeName)),
      removed: filterEntries(change.removed, (_, r) => this.scopedTypes[scope].has(r.typeName))
    };
    if (Object.keys(result.added).length === 0 && Object.keys(result.updated).length === 0 && Object.keys(result.removed).length === 0) {
      return null;
    }
    return result;
  }
  /**
   * Update the history with a diff of changes.
   *
   * @param changes - The changes to add to the history.
   */
  updateHistory(changes) {
    this.historyAccumulator.add({
      changes,
      source: this.isMergingRemoteChanges ? "remote" : "user"
    });
    if (this.listeners.size === 0) {
      this.historyAccumulator.clear();
    }
    this.history.set(this.history.get() + 1, changes);
  }
  validate(phase) {
    this.allRecords().forEach((record) => this.schema.validateRecord(this, record, phase, null));
  }
  /**
   * Add some records to the store. It's an error if they already exist.
   *
   * @param records - The records to add.
   * @public
   */
  put = (records, phaseOverride) => {
    this.atomic(() => {
      const updates = {};
      const additions = {};
      const currentMap = this.atoms.__unsafe__getWithoutCapture();
      let map = null;
      let record;
      let didChange = false;
      const source = this.isMergingRemoteChanges ? "remote" : "user";
      for (let i = 0, n = records.length; i < n; i++) {
        record = records[i];
        const recordAtom = (map ?? currentMap)[record.id];
        if (recordAtom) {
          const initialValue = recordAtom.__unsafe__getWithoutCapture();
          record = this.sideEffects.handleBeforeChange(initialValue, record, source);
          const validated = this.schema.validateRecord(
            this,
            record,
            phaseOverride ?? "updateRecord",
            initialValue
          );
          if (validated === initialValue) continue;
          recordAtom.set(devFreeze(record));
          didChange = true;
          const updated = recordAtom.__unsafe__getWithoutCapture();
          updates[record.id] = [initialValue, updated];
          this.addDiffForAfterEvent(initialValue, updated);
        } else {
          record = this.sideEffects.handleBeforeCreate(record, source);
          didChange = true;
          record = this.schema.validateRecord(
            this,
            record,
            phaseOverride ?? "createRecord",
            null
          );
          additions[record.id] = record;
          this.addDiffForAfterEvent(null, record);
          if (!map) {
            map = { ...currentMap };
          }
          map[record.id] = atom("atom:" + record.id, record);
        }
      }
      if (map) {
        this.atoms.set(map);
      }
      if (!didChange) return;
      this.updateHistory({
        added: additions,
        updated: updates,
        removed: {}
      });
    });
  };
  /**
   * Remove some records from the store via their ids.
   *
   * @param ids - The ids of the records to remove.
   * @public
   */
  remove = (ids) => {
    this.atomic(() => {
      const cancelled = /* @__PURE__ */ new Set();
      const source = this.isMergingRemoteChanges ? "remote" : "user";
      if (this.sideEffects.isEnabled()) {
        for (const id of ids) {
          const atom2 = this.atoms.__unsafe__getWithoutCapture()[id];
          if (!atom2) continue;
          if (this.sideEffects.handleBeforeDelete(atom2.get(), source) === false) {
            cancelled.add(id);
          }
        }
      }
      let removed = void 0;
      this.atoms.update((atoms) => {
        let result = void 0;
        for (const id of ids) {
          if (cancelled.has(id)) continue;
          if (!(id in atoms)) continue;
          if (!result) result = { ...atoms };
          if (!removed) removed = {};
          delete result[id];
          const record = atoms[id].get();
          removed[id] = record;
          this.addDiffForAfterEvent(record, null);
        }
        return result ?? atoms;
      });
      if (!removed) return;
      this.updateHistory({ added: {}, updated: {}, removed });
    });
  };
  /**
   * Get the value of a store record by its id.
   *
   * @param id - The id of the record to get.
   * @public
   */
  get = (id) => {
    return this.atoms.get()[id]?.get();
  };
  /**
   * Get the value of a store record by its id without updating its epoch.
   *
   * @param id - The id of the record to get.
   * @public
   */
  unsafeGetWithoutCapture = (id) => {
    return this.atoms.get()[id]?.__unsafe__getWithoutCapture();
  };
  /**
   * Creates a JSON payload from the record store.
   *
   * @param scope - The scope of records to serialize. Defaults to 'document'.
   * @returns The record store snapshot as a JSON payload.
   */
  serialize = (scope = "document") => {
    const result = {};
    for (const [id, atom2] of objectMapEntries(this.atoms.get())) {
      const record = atom2.get();
      if (scope === "all" || this.scopedTypes[scope].has(record.typeName)) {
        result[id] = record;
      }
    }
    return result;
  };
  /**
   * Get a serialized snapshot of the store and its schema.
   *
   * ```ts
   * const snapshot = store.getStoreSnapshot()
   * store.loadStoreSnapshot(snapshot)
   * ```
   *
   * @param scope - The scope of records to serialize. Defaults to 'document'.
   *
   * @public
   */
  getStoreSnapshot(scope = "document") {
    return {
      store: this.serialize(scope),
      schema: this.schema.serialize()
    };
  }
  /**
   * @deprecated use `getSnapshot` from the 'tldraw' package instead.
   */
  getSnapshot(scope = "document") {
    console.warn(
      "[tldraw] `Store.getSnapshot` is deprecated and will be removed in a future release. Use `getSnapshot` from the `tldraw` package instead."
    );
    return this.getStoreSnapshot(scope);
  }
  /**
   * Migrate a serialized snapshot of the store and its schema.
   *
   * ```ts
   * const snapshot = store.getSnapshot()
   * store.migrateSnapshot(snapshot)
   * ```
   *
   * @param snapshot - The snapshot to load.
   * @public
   */
  migrateSnapshot(snapshot) {
    const migrationResult = this.schema.migrateStoreSnapshot(snapshot);
    if (migrationResult.type === "error") {
      throw new Error(`Failed to migrate snapshot: ${migrationResult.reason}`);
    }
    return {
      store: migrationResult.value,
      schema: this.schema.serialize()
    };
  }
  /**
   * Load a serialized snapshot.
   *
   * ```ts
   * const snapshot = store.getStoreSnapshot()
   * store.loadStoreSnapshot(snapshot)
   * ```
   *
   * @param snapshot - The snapshot to load.
   * @public
   */
  loadStoreSnapshot(snapshot) {
    const migrationResult = this.schema.migrateStoreSnapshot(snapshot);
    if (migrationResult.type === "error") {
      throw new Error(`Failed to migrate snapshot: ${migrationResult.reason}`);
    }
    const prevSideEffectsEnabled = this.sideEffects.isEnabled();
    try {
      this.sideEffects.setIsEnabled(false);
      this.atomic(() => {
        this.clear();
        this.put(Object.values(migrationResult.value));
        this.ensureStoreIsUsable();
      });
    } finally {
      this.sideEffects.setIsEnabled(prevSideEffectsEnabled);
    }
  }
  /**
   * @public
   * @deprecated use `loadSnapshot` from the 'tldraw' package instead.
   */
  loadSnapshot(snapshot) {
    console.warn(
      "[tldraw] `Store.loadSnapshot` is deprecated and will be removed in a future release. Use `loadSnapshot` from the 'tldraw' package instead."
    );
    this.loadStoreSnapshot(snapshot);
  }
  /**
   * Get an array of all values in the store.
   *
   * @returns An array of all values in the store.
   * @public
   */
  allRecords = () => {
    return objectMapValues(this.atoms.get()).map((atom2) => atom2.get());
  };
  /**
   * Removes all records from the store.
   *
   * @public
   */
  clear = () => {
    this.remove(objectMapKeys(this.atoms.get()));
  };
  /**
   * Update a record. To update multiple records at once, use the `update` method of the
   * `TypedStore` class.
   *
   * @param id - The id of the record to update.
   * @param updater - A function that updates the record.
   */
  update = (id, updater) => {
    const atom2 = this.atoms.get()[id];
    if (!atom2) {
      console.error(`Record ${id} not found. This is probably an error`);
      return;
    }
    this.put([updater(atom2.__unsafe__getWithoutCapture())]);
  };
  /**
   * Get whether the record store has a id.
   *
   * @param id - The id of the record to check.
   * @public
   */
  has = (id) => {
    return !!this.atoms.get()[id];
  };
  /**
   * Add a new listener to the store.
   *
   * @param onHistory - The listener to call when the store updates.
   * @param filters - Filters to apply to the listener.
   * @returns A function to remove the listener.
   */
  listen = (onHistory, filters) => {
    this._flushHistory();
    const listener = {
      onHistory,
      filters: {
        source: filters?.source ?? "all",
        scope: filters?.scope ?? "all"
      }
    };
    this.listeners.add(listener);
    if (!this.historyReactor.scheduler.isActivelyListening) {
      this.historyReactor.start();
    }
    return () => {
      this.listeners.delete(listener);
      if (this.listeners.size === 0) {
        this.historyReactor.stop();
      }
    };
  };
  isMergingRemoteChanges = false;
  /**
   * Merge changes from a remote source without triggering listeners.
   *
   * @param fn - A function that merges the external changes.
   * @public
   */
  mergeRemoteChanges = (fn) => {
    if (this.isMergingRemoteChanges) {
      return fn();
    }
    if (this._isInAtomicOp) {
      throw new Error("Cannot merge remote changes while in atomic operation");
    }
    try {
      this.isMergingRemoteChanges = true;
      transact(fn);
    } finally {
      this.isMergingRemoteChanges = false;
    }
  };
  /**
   * Run `fn` and return a {@link RecordsDiff} of the changes that occurred as a result.
   */
  extractingChanges(fn) {
    const changes = [];
    const dispose = this.historyAccumulator.addInterceptor((entry) => changes.push(entry.changes));
    try {
      transact(fn);
      return squashRecordDiffs(changes);
    } finally {
      dispose();
    }
  }
  applyDiff(diff, {
    runCallbacks = true,
    ignoreEphemeralKeys = false
  } = {}) {
    this.atomic(() => {
      const toPut = objectMapValues(diff.added);
      for (const [_from, to] of objectMapValues(diff.updated)) {
        const type = this.schema.getType(to.typeName);
        if (ignoreEphemeralKeys && type.ephemeralKeySet.size) {
          const existing = this.get(to.id);
          if (!existing) {
            toPut.push(to);
            continue;
          }
          let changed = null;
          for (const [key, value] of Object.entries(to)) {
            if (type.ephemeralKeySet.has(key) || Object.is(value, getOwnProperty(existing, key))) {
              continue;
            }
            if (!changed) changed = { ...existing };
            changed[key] = value;
          }
          if (changed) toPut.push(changed);
        } else {
          toPut.push(to);
        }
      }
      const toRemove = objectMapKeys(diff.removed);
      if (toPut.length) {
        this.put(toPut);
      }
      if (toRemove.length) {
        this.remove(toRemove);
      }
    }, runCallbacks);
  }
  /**
   * Create a computed cache.
   *
   * @param name - The name of the derivation cache.
   * @param derive - A function used to derive the value of the cache.
   * @public
   */
  createComputedCache = (name, derive, isEqual) => {
    const cache = new WeakCache();
    return {
      get: (id) => {
        const atom2 = this.atoms.get()[id];
        if (!atom2) {
          return void 0;
        }
        return cache.get(atom2, () => {
          const recordSignal = isEqual ? computed(atom2.name + ":equals", () => atom2.get(), { isEqual }) : atom2;
          return computed(name + ":" + id, () => {
            return derive(recordSignal.get());
          });
        }).get();
      }
    };
  };
  /**
   * Create a computed cache from a selector
   *
   * @param name - The name of the derivation cache.
   * @param selector - A function that returns a subset of the original shape
   * @param derive - A function used to derive the value of the cache.
   * @public
   */
  createSelectedComputedCache = (name, selector, derive) => {
    const cache = new WeakCache();
    return {
      get: (id) => {
        const atom2 = this.atoms.get()[id];
        if (!atom2) {
          return void 0;
        }
        return cache.get(atom2, () => {
          const d = computed(
            name + ":" + id + ":selector",
            () => selector(atom2.get())
          );
          return computed(name + ":" + id, () => derive(d.get()));
        }).get();
      }
    };
  };
  _integrityChecker;
  /** @internal */
  ensureStoreIsUsable() {
    this.atomic(() => {
      this._integrityChecker ??= this.schema.createIntegrityChecker(this);
      this._integrityChecker?.();
    });
  }
  _isPossiblyCorrupted = false;
  /** @internal */
  markAsPossiblyCorrupted() {
    this._isPossiblyCorrupted = true;
  }
  /** @internal */
  isPossiblyCorrupted() {
    return this._isPossiblyCorrupted;
  }
  pendingAfterEvents = null;
  addDiffForAfterEvent(before, after) {
    assert(this.pendingAfterEvents, "must be in event operation");
    if (before === after) return;
    if (before && after) assert(before.id === after.id);
    if (!before && !after) return;
    const id = (before || after).id;
    const existing = this.pendingAfterEvents.get(id);
    if (existing) {
      existing.after = after;
    } else {
      this.pendingAfterEvents.set(id, { before, after });
    }
  }
  flushAtomicCallbacks() {
    let updateDepth = 0;
    const source = this.isMergingRemoteChanges ? "remote" : "user";
    while (this.pendingAfterEvents) {
      const events = this.pendingAfterEvents;
      this.pendingAfterEvents = null;
      if (!this.sideEffects.isEnabled()) continue;
      updateDepth++;
      if (updateDepth > 100) {
        throw new Error("Maximum store update depth exceeded, bailing out");
      }
      for (const { before, after } of events.values()) {
        if (before && after) {
          this.sideEffects.handleAfterChange(before, after, source);
        } else if (before && !after) {
          this.sideEffects.handleAfterDelete(before, source);
        } else if (!before && after) {
          this.sideEffects.handleAfterCreate(after, source);
        }
      }
      if (!this.pendingAfterEvents) {
        this.sideEffects.handleOperationComplete(source);
      }
    }
  }
  _isInAtomicOp = false;
  /** @internal */
  atomic(fn, runCallbacks = true) {
    return transact(() => {
      if (this._isInAtomicOp) {
        if (!this.pendingAfterEvents) this.pendingAfterEvents = /* @__PURE__ */ new Map();
        return fn();
      }
      this.pendingAfterEvents = /* @__PURE__ */ new Map();
      const prevSideEffectsEnabled = this.sideEffects.isEnabled();
      this.sideEffects.setIsEnabled(runCallbacks ?? prevSideEffectsEnabled);
      this._isInAtomicOp = true;
      try {
        const result = fn();
        this.flushAtomicCallbacks();
        return result;
      } finally {
        this.pendingAfterEvents = null;
        this.sideEffects.setIsEnabled(prevSideEffectsEnabled);
        this._isInAtomicOp = false;
      }
    });
  }
  /** @internal */
  addHistoryInterceptor(fn) {
    return this.historyAccumulator.addInterceptor(
      (entry) => fn(entry, this.isMergingRemoteChanges ? "remote" : "user")
    );
  }
}
function squashHistoryEntries(entries) {
  if (entries.length === 0) return [];
  const chunked = [];
  let chunk = [entries[0]];
  let entry;
  for (let i = 1, n = entries.length; i < n; i++) {
    entry = entries[i];
    if (chunk[0].source !== entry.source) {
      chunked.push(chunk);
      chunk = [];
    }
    chunk.push(entry);
  }
  chunked.push(chunk);
  return devFreeze(
    chunked.map((chunk2) => ({
      source: chunk2[0].source,
      changes: squashRecordDiffs(chunk2.map((e) => e.changes))
    }))
  );
}
class HistoryAccumulator {
  _history = [];
  _interceptors = /* @__PURE__ */ new Set();
  addInterceptor(fn) {
    this._interceptors.add(fn);
    return () => {
      this._interceptors.delete(fn);
    };
  }
  add(entry) {
    this._history.push(entry);
    for (const interceptor of this._interceptors) {
      interceptor(entry);
    }
  }
  flush() {
    const history = squashHistoryEntries(this._history);
    this._history = [];
    return history;
  }
  clear() {
    this._history = [];
  }
  hasChanges() {
    return this._history.length > 0;
  }
}
function createComputedCache(name, derive, isEqual) {
  const cache = new WeakCache();
  return {
    get(context, id) {
      const computedCache = cache.get(context, () => {
        const store = context instanceof Store ? context : context.store;
        return store.createComputedCache(name, (record) => derive(context, record), isEqual);
      });
      return computedCache.get(id);
    }
  };
}
export {
  Store,
  createComputedCache
};
//# sourceMappingURL=Store.mjs.map
